﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Runtime.Serialization.Json;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using Microsoft.Xrm.Sdk;

namespace VA.PPMS.CRM.Plugins.Helper
{

    public static class AddressValidationHelper
    {
        public static ITracingService tracingService;

        public static RootObjectResponse ValidateWC(string streetAddress, string city, string state, string zip, ITracingService _tracingService)
        {
            try
            {
                tracingService = _tracingService;

                tracingService.Trace("Beginning Address Validation");

                tracingService.Trace("Address Attribues for Request:" + streetAddress + ", " + city + ", " + state + ", " + zip);
                var addressData = new RootObjectResponse();

                using (MyClient client = new MyClient())
                {
                    client.Headers[HttpRequestHeader.ContentType] = "application/json";

                    var payload = new RootObjectRequest
                    {
                        requestAddress = new Address
                        {
                            addressLine1 = streetAddress,
                            city = city,
                            stateProvince = new StateProvince
                            {
                                name = state
                            },
                            zipCode5 = zip
                        }
                    };
                    var json = Serializee(payload);

                    tracingService.Trace("Making Call to Address Validation Servivce");
                    string response = client.UploadString(new Uri("https://int.vet360.DNS   /address-validation/address/v1/validate"), json);
                    addressData = Deserialize<RootObjectResponse>(response);
                    if (addressData != null && addressData.addressMetaData.deliveryPointValidation == "CONFIRMED")
                    {
                        tracingService.Trace("Sucessful Response from Address Validation API");                        
                    }
                    tracingService.Trace("Unsuccessful Response from Address Validation API");
                }
                return addressData;
            }
            catch (WebException we)
            {
                tracingService.Trace("Web Exception on call to Address Validation Service:", we.Message.ToString());
                throw;
            }
            catch (Exception e)
            {
                tracingService.Trace("Exception on call to Address Validation Service:", e.ToString());
                return null;
            }
        }

        public static AddressData ValidateWcDWS(string streetAddress, string city, string state, string zip, ITracingService _tracingService)
        {
            try
            {
                tracingService = _tracingService;
                tracingService.Trace("Beginning Address Validation");
                tracingService.Trace("Address Attribues for Request:" + streetAddress + ", " + city + ", " + state + ", " + zip);

                var addressData = new AddressData();

                using (WebClient wc = new WebClient())
                {
                    //string addressString = "http://dev.dws.ppms.DNS   /v1.0/ValidateAddress?streetAddress=";
                    string proxyAddressString = "https://dev.integration.d365.DNS   /ase/v1.0/ValidateAddress?streetAddress=";

                    proxyAddressString += "'" + streetAddress + "'";
                    proxyAddressString += "&city=";
                    proxyAddressString += city;
                    proxyAddressString += "&state=";
                    proxyAddressString += state;
                    proxyAddressString += "&zip=";
                    proxyAddressString += zip;
                    proxyAddressString = Uri.EscapeUriString(proxyAddressString).Replace("#", "%23");

                    //var testString = "https://dev.dws.ppms.DNS   /v1.0/ValidateAddress?streetAddress=";

                    tracingService.Trace("Address String: " + proxyAddressString);

                    wc.Headers[HttpRequestHeader.ContentType] = "application/json";

                    tracingService.Trace("Making Call to DWS Address Validation Function");
                    string dwsResponse = wc.DownloadString(proxyAddressString);
                    AddressDataValues addressDataValues = Deserialize<AddressDataValues>(dwsResponse);
                    if (addressDataValues.value.Count > 0)
                    {
                        tracingService.Trace("Sucessful Response from Address Validation API");                        
                        addressData = addressDataValues.value[0];
                        _tracingService.Trace("Confidence Score:" + addressData.ConfidenceScore);
                    }
                    else
                    {
                        tracingService.Trace("Unsuccessful Response from Address Validation API");
                    }
                    return addressData;
                }                           
            }
            catch (WebException we)
            {
                tracingService.Trace("Web Exception on call to Address Validation Service:", we.Message.ToString());
                return null;
            }
            catch (Exception e)
            {
                tracingService.Trace("Exception on call to Address Validation Service:", e.ToString());
                return null;
            }
        }

        public static AddressData ValidateHttp(string streetAddress, string city, string state, string zip, ITracingService _tracingService)
        {
            try
            {
                tracingService = _tracingService;
                tracingService.Trace("Beginning Address Validation");
                tracingService.Trace("Address Attribues for Request:" + streetAddress + ", " + city + ", " + state + ", " + zip);

                var addressData = new AddressData();

                using (HttpClient client = new HttpClient())
                {
                    //string addressString = "https://dev.dws.ppms.DNS   /v1.0/ValidateAddress?streetAddress=";
                    string proxyAddressString = "https://dev.integration.d365.DNS   /ase/v1.0/ValidateAddress?streetAddress=";

                    proxyAddressString += "'1239 Vermont Ave NW #605'";
                    proxyAddressString += "&city=";
                    proxyAddressString += "Washington";
                    proxyAddressString += "&state=";
                    proxyAddressString += "DC";
                    proxyAddressString += "&zip=";
                    proxyAddressString += "20005";
                    proxyAddressString = Uri.EscapeUriString(proxyAddressString).Replace("#", "%23");

                    tracingService.Trace("Making Call to DWS Address Validation Function");

                    var response = client.GetStringAsync(proxyAddressString).GetAwaiter().GetResult();

                    AddressDataValues addressDataValues = Deserialize<AddressDataValues>(response);
                    if (addressDataValues.value.Count > 0)
                    {
                        tracingService.Trace("Sucessful Response from Address Validation API");
                        addressData = addressDataValues.value[0];
                    }
                }
                tracingService.Trace("Unsuccessful Response from Address Validation API");
                return addressData;
            }
            catch (WebException we)
            {
                tracingService.Trace("Web Exception on call to Address Validation Service:", we.Message.ToString());
                throw;
            }
            catch (Exception e)
            {
                tracingService.Trace("Exception on call to Address Validation Service:", e.ToString());
                throw;
            }
        }

        public static RootObjectResponse Validate(string streetAddress, string city, string state, string zip, Microsoft.Xrm.Sdk.ITracingService tracingService)
        {
            try
            {
                tracingService.Trace("Beginning Address Validation");
                //ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
                //ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;
                //ServicePointManager.SecurityProtocol = SecurityProtocolType.Ssl3 | SecurityProtocolType.Tls | SecurityProtocolType.Tls11 | SecurityProtocolType.Tls12;

                tracingService.Trace("Address Attribues for Request:" + streetAddress + ", " + city + ", " + state + ", " + zip);
                var addressData = new RootObjectResponse();

                using (var client = GetHttpClient(tracingService))
                {
                    tracingService.Trace("Setting Client Base Address");

                    

                    client.BaseAddress = new Uri("https://int.vet360.DNS   ");                   
                    var payload = new RootObjectRequest
                    {
                        requestAddress = new Address
                        {
                            addressLine1 = streetAddress,
                            city = city,
                            stateProvince = new StateProvince
                            {
                                name = state
                            },
                            zipCode5 = zip
                        }
                    };

                    var json = Serializee(payload);

                    var content = new StringContent(json, Encoding.UTF8, "application/json");
                    content.Headers.ContentType = new MediaTypeHeaderValue("application/json");

                    tracingService.Trace("Making Call to Address Validation Servivce");
                    var response = client.PostAsync("/address-validation/address/v1/validate", content).GetAwaiter().GetResult();

                    if (response.IsSuccessStatusCode)
                    {
                        tracingService.Trace("Sucessful Response from Address Validation API");
                        var result = response.Content.ReadAsStringAsync().GetAwaiter().GetResult();
                        if (!string.IsNullOrEmpty(result))
                        {              
                            addressData = Deserialize<RootObjectResponse>(result);                           
                        }
                    }
                    tracingService.Trace("Unsuccessful Response from Address Validation API");
                }
                return addressData;
            }
            catch (HttpRequestException he)
            {
                tracingService.Trace("Http Web Exception on call to Address Validation Service:", he.ToString());               
                return null;
            }
            catch (Exception e)
            {
                tracingService.Trace("Exception on call to Address Validation Service:", e.ToString());
                throw;
            }    
        }

        public static string Serializee<T>(T data)
        {
            var ms = new MemoryStream();
            var ser = new DataContractJsonSerializer(typeof(T));
            ser.WriteObject(ms, data);
            var json = ms.ToArray();
            ms.Close();

            return Encoding.UTF8.GetString(json, 0, json.Length);
        }

        public static T Deserialize<T>(string json)
        {
            var ms = new MemoryStream(Encoding.UTF8.GetBytes(json));
            var ser = new DataContractJsonSerializer(typeof(T));
            var result = (T) ser.ReadObject(ms);
            ms.Close();

            return result;
        }

        private static HttpClient GetHttpClient(ITracingService tracingService)
        {
            try
            {
                tracingService.Trace("Getting HttpClient");
                var clientHandler = new WebRequestHandler();
                clientHandler.ClientCertificates.Add(GetCertKeyVault(tracingService));
                //clientHandler.ClientCertificates.Add(GetLocalCert());
                tracingService.Trace("Client Handler Established");
                return new HttpClient(clientHandler);
            }
            catch(WebException we)
            {
                tracingService.Trace("Client Handler Web Exception");
                throw;
            }
            catch(Exception e)
            {
                tracingService.Trace("Client Handler Exception");
                throw;
            }
        }

        class MyClient : WebClient
        {
            protected override WebRequest GetWebRequest(Uri address)
            {
                HttpWebRequest request = (HttpWebRequest)base.GetWebRequest(address);
                request.ClientCertificates.Add(GetCertKeyVault(tracingService));
                return request;
            }
        }

        public static X509Certificate2 GetLocalCert()
        {
            var store = new X509Store(StoreName.My, StoreLocation.LocalMachine);
            store.Open(OpenFlags.OpenExistingOnly | OpenFlags.ReadOnly);
            string certificateSubjectName = "CN=dws.ppms.DNS   , OU=PPMS, O=VA, L=Washington, S=DC, C=US";
            var cert = store.Certificates.Find(X509FindType.FindBySubjectDistinguishedName, certificateSubjectName,
                true);
            if (cert.Count < 1)
            {
                throw new Exception(string.Format("Could not find a valid client certificate with subject {0}",
                    certificateSubjectName));
            }

            return cert[0];
        }

        private static X509Certificate2 GetCertKeyVault(Microsoft.Xrm.Sdk.ITracingService tracingService)
        {
            try
            {
                tracingService.Trace("Retrieve Cert from Key Vault");

                const string appId = "REDACTED";
                const string secret = "REDACTED";
                const string tenantId = "REDACTED";
                const string certUri = "https://vanpeastppmskv1.vault.usgovcloudapi.net/certificates/np-dws-ppms-nprod/REDACTED";

                var token = GetToken(appId, PW    , tenantId, tracingService);
                if (token != null) { tracingService.Trace("Token received"); }
                var cert = GetCertificateFromKeyVault(token.access_token, certUri, tracingService);
                if (cert != null) { tracingService.Trace("Cert received"); }
                var privateKey = GetPrivateKeyKeyVault(token.access_token, cert.sid, tracingService);
                if (privateKey != null) { tracingService.Trace("Private Key received"); }

                var x509Cert = new X509Certificate2(privateKey);
                if(x509Cert != null) { tracingService.Trace("Cert Created Successfully"); }
                else { tracingService.Trace("Cert Creation Failure"); }

                return x509Cert;
            }
            catch (WebException we)
            {
                tracingService.Trace("GetCertKeyVault WebException");
                return null;
            }
            catch(Exception e)
            {
                tracingService.Trace("GetCertKeyVault Exception: " +  e.Message.ToString());
                return null;
            }
        }

        private static TokenResponse GetToken(string clientId, string clientSecret, string tenantId, Microsoft.Xrm.Sdk.ITracingService tracingService)
        {
            try
            {
                tracingService.Trace("Get Token");
                using (var httpClient = new HttpClient())
                {
                    var formContent = new FormUrlEncodedContent(new[]
                    {
                    new KeyValuePair<string, string>("resource", "https://vault.usgovcloudapi.net"),
                    new KeyValuePair<string, string>("client_id", clientId),
                    new KeyValuePair<string, string>("client_secret", clientSecret),
                    new KeyValuePair<string, string>("grant_type", "client_credentials")
                });

                    var response = httpClient
                        .PostAsync("https://login.windows.net/" + tenantId + "/oauth2/token", formContent).GetAwaiter()
                        .GetResult();

                    return Deserialize<TokenResponse>(response.Content.ReadAsStringAsync().Result);
                }
            }
            catch(WebException we)
            {
                tracingService.Trace("GetToken WebException");
                return null;
            }
            catch(Exception e)
            {
                tracingService.Trace("GetToken Exception");
                return null;
            }
        }

        public static CertificateResponse GetCertificateFromKeyVault(string token, string certificateUrl, Microsoft.Xrm.Sdk.ITracingService tracingService)
        {
            try
            {
                using (var httpClient = new HttpClient())
                {
                    var request =
                        new HttpRequestMessage(HttpMethod.Get, new Uri(certificateUrl + "?api-version=2016-10-01"));
                    request.Headers.Authorization = new AuthenticationHeaderValue("Bearer", token);

                    var response = httpClient.SendAsync(request).GetAwaiter().GetResult();
                    if (response.IsSuccessStatusCode) { tracingService.Trace("Successful Response retrieving Certficate from KeyVault"); }

                    return Deserialize<CertificateResponse>(response.Content.ReadAsStringAsync().GetAwaiter().GetResult());
                }
            }
            catch(WebException we)
            {
                tracingService.Trace("GetCertificateFromKeyVault WebException");
                return null;
            }
            catch(Exception e)
            {
                tracingService.Trace("GetCertificateFromKeyVault Exception");
                return null;
            }
        }
        public static byte[] GetPrivateKeyKeyVault(string token, string certificateUrl, Microsoft.Xrm.Sdk.ITracingService tracingService)
        {
            try
            {
                using (var httpClient = new HttpClient())
                {
                    var request = new HttpRequestMessage(HttpMethod.Get, new Uri(certificateUrl + "?api-version=2016-10-01"));
                    request.Headers.Authorization = new AuthenticationHeaderValue("Bearer", token);
                    var response = httpClient.SendAsync(request).GetAwaiter().GetResult();

                    var privateKey = Deserialize<PrivateKeyResponse>(response.Content.ReadAsStringAsync().GetAwaiter().GetResult());
                    if (response.IsSuccessStatusCode) { tracingService.Trace("Successful Response retrieving Private Key"); }
                    return Convert.FromBase64String(privateKey.value);
                }
            }
            catch(WebException we)
            {
                tracingService.Trace("GetPrivateKeyKeyVault WebException");
                return null;
            }
            catch(Exception e)
            {
                tracingService.Trace("GetPrivateKeyKeyVault Exception");
                return null;
            }
        }
    }

    public class TokenResponse
    {
        public string token_type { get; set; }
        public string expires_in { get; set; }
        public string ext_expires_in { get; set; }
        public string expires_on { get; set; }
        public string not_before { get; set; }
        public string resource { get; set; }
        public string access_token { get; set; }
    }

    public class CertificateResponse
    {
        public string id { get; set; }
        public string kid { get; set; }
        public string sid { get; set; }
        public string x5t { get; set; }
        public string cer { get; set; }
    }
    public class PrivateKeyResponse
    {
        public string value { get; set; }
    }

}
